%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Function name: apply_ch4_spec_int_corr
%
% Scope:    apply CH4 spectral interference correction
%
% Input:    data (structure),
%           bindata (structure),
%           binref (structure),
%           list of gas names (cell),
%           list of gas type "flags" (i.e., MPV positions) (double),
%           slope spectral interference correction (double)
%           anchor gas name (string),
%           index plot requested (logical)
%
% Output:   corrected "bindata" and "binref" structures,
%           overview figures (if requested)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [ref, binref, bindata] = apply_ch4_spec_int_corr(data, bindata, binref, gasnames, gastypeflag, slope_ch4, anchor_gas, plot_requested)
    
tic;fprintf('Running function apply_ch4_spec_int_corr...\n');

bindata_idx=2;     %N2O index
bindata_idx2=12;   %CH4 index
target_idx=4:6;    %d15N, d15Na, d15Nb index

ind_anchor=find(ismember(gasnames,anchor_gas));

% calculate intercept for given slopes (only needed for plotting, not used in correction)
y_anchor=NaN(1,width(target_idx));
intercept_ch4=NaN(1,width(target_idx));
for z=1:width(target_idx)
    y_anchor(1,z) = slope_ch4(1,z).*(binref.(gasnames{1,ind_anchor})(1,bindata_idx2)./binref.(gasnames{1,ind_anchor})(1,bindata_idx));
    intercept_ch4(1,z)=binref.(gasnames{1,ind_anchor})(1,target_idx(1,z)) - y_anchor(1,z);
end

axislabel={'CH_4 / N_2O [ppm ppb^{-1}]'};

%% plot data before correction

if plot_requested
    plot_ch4_spec_int_corr(data,bindata,binref,gasnames,bindata_idx,bindata_idx2,target_idx,slope_ch4,intercept_ch4,axislabel,ind_anchor,'Before CH_4 spectral interference correction');
end

%% apply correction

%correct data
for j=1:width(target_idx)
    for i=1:width(gastypeflag)
           bindata.(gasnames{1,i})(:,target_idx(1,j))=bindata.(gasnames{1,i})(:,target_idx(1,j))-slope_ch4(1,j)*((bindata.(gasnames{1,i})(:,bindata_idx2)./bindata.(gasnames{1,i})(:,bindata_idx))-(binref.(gasnames{1,ind_anchor})(1,bindata_idx2)./binref.(gasnames{1,ind_anchor})(1,bindata_idx)));
    end
    bindata.all(:,target_idx(1,j))=bindata.all(:,target_idx(1,j))-slope_ch4(1,j)*((bindata.all(:,bindata_idx2)./bindata.all(:,bindata_idx))-(binref.(gasnames{1,ind_anchor})(1,bindata_idx2)./binref.(gasnames{1,ind_anchor})(1,bindata_idx)));
end

%% recalculate ref conditions

[ref, binref, bindata] = calc_ref_conditions(data,bindata,gasnames,gastypeflag);

%% plot data after correction

if plot_requested
    plot_ch4_spec_int_corr(data,bindata,binref,gasnames,bindata_idx,bindata_idx2,target_idx,slope_ch4,intercept_ch4,axislabel,ind_anchor,'After CH_4 spectral interference correction');
end

%%
time_elapsed=toc; fprintf('apply_ch4_spec_int_corr completed (execution time: %1.2f s)\n',time_elapsed); 

end

